﻿using System;
using System.Collections.Generic;
using System.Text;

using Thomas_Erdoesi.Game_Objects;
using Thomas_Erdoesi.Properties;

namespace Thomas_Erdoesi.Game_Analyzer
{
    class FrameAnalyzer
    {
        /// <summary>
        /// Übermittelten Frame analysieren
        /// </summary>
        /// <param name="Status"></param>
        public unsafe void Analyze(GameStatus Status)
        {
            int dx;
            int dy;
            int sf;
            int vx = 0;
            int vy = 0, vz = 0, vs = 0;
            int v1x = 0;
            int v1y = 0;
            int shipdetect = 0;

            Status.PredictLocation();
            Status.PrepareStatusUpdate();

            for (int pc = 1; pc < 513; )
            {
                int op = Status.Frame.vectorram[pc] >> 12; ;
                switch (op)
                {
                    case 0xa: // LABS
                        vy = Status.Frame.vectorram[pc] & 0x3ff;
                        vx = Status.Frame.vectorram[pc + 1] & 0x3ff;
                        vs = Status.Frame.vectorram[pc + 1] >> 12;
                        break;

                    case 0xb: // HALT
                        MatchShots(Status);

                        Status.Score = int.Parse(Status.Score).ToString();
                        return;

                    case 0xc: // JSRL
                        ProcessJSRL(Status, vx, vy, vs, Status.Frame.vectorram[pc] & 0xfff);
                        break;

                    case 0xd: // RTSL
                        return;

                    case 0xe: // JMPL
                        return;

                    case 0xf: // SVEC
                        break;

                    default:
                        dy = Status.Frame.vectorram[pc] & 0x3ff;
                        if ((Status.Frame.vectorram[pc] & 0x400) != 0)
                            dy = -dy;
                        dx = Status.Frame.vectorram[pc + 1] & 0x3ff;
                        if ((Status.Frame.vectorram[pc + 1] & 0x400) != 0)
                            dx = -dx;
                        sf = op;
                        vz = Status.Frame.vectorram[pc + 1] >> 12;
                        if (dx == 0 && dy == 0 && vz == 15)
                            AddShot(Status,vx, vy);
                        if (op == 6 && vz == 12 && dx != 0 && dy != 0)
                        {
                            switch (shipdetect)
                            {
                                case 0:
                                    v1x = dx;
                                    v1y = dy;
                                    ++shipdetect;
                                    break;
                                case 1:
                                    AddSpaceship(Status, vx, vy, v1x - dx, v1y - dy);
                                    ++shipdetect;
                                    break;
                            }
                        }
                        else if (shipdetect == 1)
                            shipdetect = 0;

                        break;
                }

                if (op <= 0xa)
                    ++pc;

                if (op != 0xe) // JMPL
                    ++pc;
            }
        }

        /// <summary>
        /// Mit Hilfe von JSRL realisierte Objekte analysieren
        /// </summary>
        /// <param name="Status"></param>
        /// <param name="vx"></param>
        /// <param name="vy"></param>
        /// <param name="vs"></param>
        /// <param name="Address"></param>
        public void ProcessJSRL(GameStatus Status, int vx, int vy, int vs, int Address)
        {
            int Size = 0;

            switch (vs)
            {
                case 0:  
                    Size = 40;
                    break;

                case 15: 
                    Size = 20;
                    break;

                case 14: 
                    Size = 8;
                    break;

                default:
                    Size = 10;
                    break;
            }

            switch (Address)
            {
                case 0x8f3:
                    AddAsteroid(Status, vx, vy, Size, 1);
                    break;

                case 0x8ff:
                    AddAsteroid(Status, vx, vy, Size, 2);
                    break;

                case 0x90d:
                    AddAsteroid(Status, vx, vy, Size, 3);
                    break;

                case 0x91a:
                    AddAsteroid(Status, vx, vy, Size, 4);
                    break;

                case 0x929:
                    AddSaucer(Status, vx, vy, Size);
                    break;

                default:
                    if (vs == 1 && vy == 876 && Status.Score.Length < 5) ParseScore(Status, Address);
                    break;
            }
        }

        /// <summary>
        /// Aktuellen Punktestand parsen
        /// </summary>
        /// <param name="Status"></param>
        /// <param name="Address"></param>
        public void ParseScore(GameStatus Status, int Address)
        {
            switch (Address)
            {
                case 2860:
                    Status.Score = Status.Score + " ";
                    break;

                case 2781:
                    Status.Score = Status.Score + "0";
                    break;

                case 2862:
                    Status.Score = Status.Score + "1";
                    break;

                case 2866:
                    Status.Score = Status.Score + "2";
                    break;

                case 2874:
                    Status.Score = Status.Score + "3";
                    break;

                case 2881:
                    Status.Score = Status.Score + "4";
                    break;

                case 2888:
                    Status.Score = Status.Score + "5";
                    break;

                case 2895:
                    Status.Score = Status.Score + "6";
                    break;

                case 2902:
                    Status.Score = Status.Score + "7";
                    break;

                case 2907:
                    Status.Score = Status.Score + "8";
                    break;

                case 2915:
                    Status.Score = Status.Score + "9";
                    break;
            }
        }

        /// <summary>
        /// Asteroid hinzufügen
        /// </summary>
        /// <param name="Status"></param>
        /// <param name="vx"></param>
        /// <param name="vy"></param>
        /// <param name="Size"></param>
        /// <param name="Type"></param>
        public void AddAsteroid(GameStatus Status, int vx, int vy, int Size, int Type)
        {
            Asteroid ExistingAsteroid = null;

            double MinDist = Settings.Default.PredictionMatchingThresholdAsteroid;

            foreach (Asteroid PrevAsteroid in Status.PrevAsteroids)
            {
                if (PrevAsteroid.Size == Size && PrevAsteroid.Type == Type)
                {
                    GameObjectDistance Distance = PrevAsteroid.Location.GetDistance(new GameObjectLocation(vx, vy));

                    if (Distance.CenterDistance < MinDist)
                    {
                        MinDist = Distance.CenterDistance;
                        ExistingAsteroid = PrevAsteroid;
                    }
                }
            }

            if (ExistingAsteroid != null)
            {
                Status.Asteroids.Add(ExistingAsteroid);
                Status.PrevAsteroids.Remove(ExistingAsteroid);

                ExistingAsteroid.Location.x = vx;
                ExistingAsteroid.Location.y = vy;

                ExistingAsteroid.IncomingShots = 0;

                if (ExistingAsteroid.Speed.sx == 0 && ExistingAsteroid.Speed.sy == 0 ||
                    Status.FrameCounter - ExistingAsteroid.DetectionFrame <= Settings.Default.AsteroidSpeedDetectionPeriod)
                {
                    GameObjectDistance Distance = ExistingAsteroid.DetectionLocation.GetDistance(ExistingAsteroid.Location);

                    ExistingAsteroid.Speed.sx = Distance.dx / (Status.FrameCounter - ExistingAsteroid.DetectionFrame);
                    ExistingAsteroid.Speed.sy = Distance.dy / (Status.FrameCounter - ExistingAsteroid.DetectionFrame);
                }
            }
            else
            {
                Status.Asteroids.Add(new Asteroid(Status, vx, vy, Size, Type));
            }
        }

        /// <summary>
        /// Schuss hinzufügen
        /// </summary>
        /// <param name="Status"></param>
        /// <param name="vx"></param>
        /// <param name="vy"></param>
        public void AddShot(GameStatus Status, int vx, int vy)
        {
            Status.Shots.Add(new Shot(Status, vx, vy));
        }

        /// <summary>
        /// Raumschiff hinzufügen
        /// </summary>
        /// <param name="Status"></param>
        /// <param name="vx"></param>
        /// <param name="vy"></param>
        /// <param name="dx"></param>
        /// <param name="dy"></param>
        public void AddSpaceship(GameStatus Status, int vx, int vy, int dx, int dy)
        {
            Status.Spaceship = new Spaceship(vx, vy);

            if (Status.PrevSpaceshipLocation != null)
            {
                GameObjectDistance Distance = Status.PrevSpaceshipLocation.GetDistance(Status.Spaceship.Location);

                Status.Spaceship.Speed.sx = Distance.dx / (Status.FrameCounter - Status.PrevFrameCounter);
                Status.Spaceship.Speed.sy = Distance.dy / (Status.FrameCounter - Status.PrevFrameCounter);
            }

            unchecked
            {
                if (Spaceship.TurnLeftExecuted) Status.Spaceship.AngleByte++;
                if (Spaceship.TurnRightExecuted) Status.Spaceship.AngleByte--;
            }

            Console.WriteLine(Status.Spaceship.AngleByte);

            Status.Spaceship.Angle = Math.Atan2(dy, dx);
        }

        /// <summary>
        /// UFO hinzufügen
        /// </summary>
        /// <param name="Status"></param>
        /// <param name="vx"></param>
        /// <param name="vy"></param>
        /// <param name="size"></param>
        public void AddSaucer(GameStatus Status, int vx, int vy, int size)
        {
            if (Status.PrevSaucer != null)
            {
                Status.Saucer = new Saucer(vx, vy, size);

                GameObjectDistance Distance = Status.PrevSaucer.Location.GetDistance(Status.Saucer.Location);

                Status.Saucer.Speed.sx = Distance.dx / (Status.FrameCounter - Status.PrevFrameCounter);
                Status.Saucer.Speed.sy = Distance.dy / (Status.FrameCounter - Status.PrevFrameCounter);
            }
            else
            {
                Status.Saucer = new Saucer(vx, vy, size);
            }
        }

        /// <summary>
        /// Bekannte Schüsse mit neuen Schüssen abgleichen
        /// </summary>
        /// <param name="Status"></param>
        public void MatchShots(GameStatus Status)
        {
            GameObjectLocation InitialShotLocation = null;

            if (Status.PrevSpaceshipLocation != null)
                InitialShotLocation = GameMath.CalculateInitialShotLocation(Status.PrevSpaceshipLocation, Status.PrevSpaceshipAngle);

            // Erkannte Schüsse mit bekannten Objekten abgleichen
            foreach (Shot CurrentShot in Status.Shots)
            {
                Shot MatchingPrevShot = null;

                double MinObjDist = Settings.Default.PredictionMatchingThresholdShot;

                foreach (Shot PrevShot in Status.PrevShots)
                {
                    double ObjDist = CurrentShot.GetDistance(PrevShot.PredictedLocation).CenterDistance;

                    if (ObjDist < MinObjDist)
                    {
                        if (Math.Abs(PrevShot.GetDistance(CurrentShot.Location).dx) < Settings.Default.MaxShotSpeed &&
                            Math.Abs(PrevShot.GetDistance(CurrentShot.Location).dy) < Settings.Default.MaxShotSpeed)
                        {
                            MinObjDist = ObjDist;
                            MatchingPrevShot = PrevShot;
                        }
                    }
                }

                if (MatchingPrevShot == null)
                {
                    // Eigenen Schüssen gleich den richtigen Geschwindigkeitsvektor mitgeben
                    if (Status.PrevSpaceshipLocation != null)
                        if (new GameObjectDistance(InitialShotLocation, CurrentShot.Location).CenterDistance <= Settings.Default.InitialShotMatchingThreshold)
                            CurrentShot.Speed = GameMath.CalculateShotSpeed(Status.PrevSpaceshipSpeed, Status.PrevSpaceshipAngle);
                }
                else 
                {
                    // Geschwindigkeit aufgrund bestehender Daten berechnen
                    CurrentShot.DetectionLocation = MatchingPrevShot.DetectionLocation;
                    CurrentShot.DetectionFrame = MatchingPrevShot.DetectionFrame;

                    CurrentShot.Speed = MatchingPrevShot.Speed;

                    if (CurrentShot.Speed.sx == 0 && CurrentShot.Speed.sy == 0 ||
                        Status.FrameCounter - CurrentShot.DetectionFrame <= Settings.Default.ShotSpeedDetectionPeriod)
                    {
                        GameObjectDistance Distance = CurrentShot.DetectionLocation.GetDistance(CurrentShot.Location);

                        CurrentShot.Speed.sx = Distance.dx / (Status.FrameCounter - CurrentShot.DetectionFrame);
                        CurrentShot.Speed.sy = Distance.dy / (Status.FrameCounter - CurrentShot.DetectionFrame);
                    }

                    Status.PrevShots.Remove(MatchingPrevShot);
                }

                // Prüfen, ob Schuss ein Ziel treffen wird
                double MinTTC = double.MaxValue;
                Asteroid ShotTarget = null;

                foreach (Asteroid TargetObject in Status.Asteroids)
                {
                    double TTC = GameMath.CalculateTimeToCollision(CurrentShot.Location, CurrentShot.Speed, CurrentShot.Size,
                                                                   TargetObject.Location, TargetObject.Speed, TargetObject.Size);

                    if (TTC < MinTTC)
                    {
                        ShotTarget = TargetObject;
                        MinTTC = TTC;
                    }
                }

                if (ShotTarget != null)
                {
                    if (CurrentShot.Speed.Speed * (MinTTC + Status.FrameCounter - CurrentShot.DetectionFrame) <= Settings.Default.EstimatedShotRange)
                        ShotTarget.IncomingShots++;
                }
            }
        }
    }
}
